package com.cku.oa.show.service;

import java.util.Calendar;
import java.util.List;
import java.util.Objects;

import javax.validation.Validator;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.entity.DogTypeShow;
import com.cku.oa.dog.service.DogTypeShowService;
import com.cku.oa.show.dao.ShowResultsHkDao;
import com.cku.oa.show.entity.ShowResults;
import com.cku.oa.show.entity.ShowResultsHk;
import com.cku.oa.show.entity.ShowScoreRanking;
import com.thinkgem.jeesite.common.beanvalidator.BeanValidators;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.common.utils.excel.ImportExcel;

/**
 * 香港赛事成绩表Service
 * @author xx
 * @version 2024-11-14
 */
@Service
@Transactional(readOnly = true)
public class ShowResultsHkService extends CrudService<ShowResultsHkDao, ShowResultsHk> {

	@Autowired
	private DogTypeShowService dogTypeShowService;
	
	@Autowired
	private ShowScoreRankingService showScoreRankingService;
	
	@Autowired
	private Validator validator;

	public ShowResultsHk get(String id) {
		return super.get(id);
	}
	
	public List<ShowResultsHk> findList(ShowResultsHk showResultsHk) {
		return super.findList(showResultsHk);
	}
	
	public Page<ShowResultsHk> findPage(Page<ShowResultsHk> page, ShowResultsHk showResultsHk) {
		return super.findPage(page, showResultsHk);
	}
	
	@Transactional(readOnly = false)
	public void save(ShowResultsHk showResultsHk) {
		preSave(showResultsHk);
		super.save(showResultsHk);
	}
	
	@Transactional(readOnly = false)
	public void delete(ShowResultsHk showResultsHk) {
		super.delete(showResultsHk);
	}
	
	
	public void preSave(ShowResultsHk showResultsHk) {
		// 积分归属年份为比赛日期所属年份
		String year = "";
		if(!Objects.isNull(showResultsHk.getStartTime())){
			Calendar calendar = Calendar.getInstance();
		    calendar.setTime(showResultsHk.getStartTime());
			year = calendar.get(Calendar.YEAR)+"";
		}
		showResultsHk.setYear(year);
		// 犬种中文
		if(StringUtils.isNotBlank(showResultsHk.getFciCode())){
			DogTypeShow dogType = dogTypeShowService.getByDogShowCode(showResultsHk.getFciCode());
			if(Objects.isNull(dogType)) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "fic编码" + 
						showResultsHk.getFciCode() +  "没有查到！请确认是否正确！");
			}
			showResultsHk.setDogBreedCn(dogType.getShowCnName());
			showResultsHk.setDogBreedEn(dogType.getShowEnName());
		}
	}
	
	/**
	 *
	 * @description: 导入犬展成绩
	 */
	@Transactional(readOnly = false, rollbackFor = Exception.class)
	public int importShowResults(MultipartFile file) throws Exception {
		int count = 0;
		try {
			// 从第2行开始读取，否则会少读一行
			ImportExcel ei = new ImportExcel(file, 0, 0);
			List<ShowResultsHk> list = ei.getDataList(ShowResultsHk.class);
			boolean flag = true;
			for (int i = 0; i < list.size(); i++) {
				ShowResultsHk showResultsHK = list.get(i);
				// 跳过无效记录
				if (StringUtils.isBlank(showResultsHK.getShowCode())) {
					continue;
				}
				flag = false;
				preSave(showResultsHK);
				// 参数校验
				try {
					BeanValidators.validateWithException(validator, showResultsHK);
				} catch (Exception e) {
					logger.warn("校验参数出错", e);
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "参数有误，请修改后再导入！");
				}
				// 判断是否已有成绩
				if (checkResultByShowCodeAndPedigreeCertified(showResultsHK.getShowCode(),
						showResultsHK.getPedigreeCertified())) {
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "已有血统证书号为" + showResultsHK.getPedigreeCertified()
							+ ",犬展编号为" + showResultsHK.getShowCode() + "的成绩记录");
				}
				// 写数据库
				try {
					showResultsHK.preInsert();
					showResultsHK.setIsNewRecord(true);
					super.save(showResultsHK);
				} catch (Exception e) {
					logger.error("第" + (i + 1) + "行数据库保存犬展成绩出错：", e);
					throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "第" + (i + 1) + "行" + "参数有误，无法入库，请修改后再导入！");
				}

				count++;
			}
			if (flag) {
				throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "未找到有效记录！");
			}
		} catch (ZAException e) {
			throw e;
		} catch (Exception e) {
			logger.warn("犬展成绩导入出错！", e);
			throw new Exception(e);
		}
		return count;
	}
	
	/**
	 * @Description 判断是否已有成绩
	 */
	public boolean checkResultByShowCodeAndPedigreeCertified(String showCode, String pedigreeCertified) {
		return dao.countByShowCodeAndPedigreeCertified(showCode, pedigreeCertified)> 0;
	}

	public List<ShowResultsHk> findAllYear(String yearLike) {
		return dao.findAllYear(yearLike);
	}
	
	@Transactional(readOnly = false, rollbackFor = Exception.class)
	public void rankingByYear(String year) {
		String scoreType = "HK";
		// 逻辑删除本年旧数据
		showScoreRankingService.delLogical(year,scoreType);
		//拿出比赛时间最晚的一场比赛的编号作为版本号
		String newShowCode =  dao.getNewShowCode(year);
		//把每条的可获分数写入
		dao.countShowResultByYear(year);
		//计算出当前排名、分数、证书号
		List<ShowResultsHk> list = dao.rankingByYear(year);
		//其他信息以记录中最新的一条为准
		int ranking = 1;
		for(int i = 0; i < list.size(); i++) {
			ShowResultsHk showResultsHk = list.get(i);
			ShowResultsHk entry = dao.getNewOne(showResultsHk.getPedigreeCertified());
			//存榜单版本数据
			ShowScoreRanking entity = new ShowScoreRanking();
			entity.setPedigreeCertified(showResultsHk.getPedigreeCertified());
			entity.setDogOwner(entry.getDogOwnerName());
			entity.setDogBreedNameCn(entry.getDogBreedCn());
			entity.setDogBreedNameEn(entry.getDogBreedEn());
			entity.setDogBreed(entry.getFciCode());
			entity.setDogName(entry.getDogName());
			entity.setYear(year);
			entity.setScoreType(scoreType);
			entity.setScore(Double.valueOf(showResultsHk.getShowResult()));
			// 排名
			entity.setRanking(ranking);
			// 版本号
			entity.setVersion(newShowCode);
			entity.preInsert();
			entity.setIsNewRecord(true);
			showScoreRankingService.save(entity);
			if (i < list.size() - 1) {
				ShowResultsHk showResultsHk2 = list.get(i + 1);
				if (!StringUtils.equals(showResultsHk.getShowResult(), showResultsHk2.getShowResult())) {
					ranking++;
				}
			}
		}

		// 上升趋势 0持平 1上升 2下降
		showScoreRankingService.saveTrend(year,scoreType);
		// 对多余的冗余数据进行物理删除
		showScoreRankingService.delPhysical(year, newShowCode, scoreType);
	
	}
	
	
}