package com.cku.oa.dog.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.cku.core.ZAErrorCode;
import com.cku.core.ZAException;
import com.cku.oa.dog.dao.DogDnaArchiveReissueSamplingDao;
import com.cku.oa.dog.entity.DogDnaArchive;
import com.cku.oa.dog.entity.DogDnaArchiveReissueSampling;
import com.cku.oa.finance.entity.PaymentChargingItem;
import com.cku.oa.finance.entity.PaymentOrder;
import com.cku.oa.finance.entity.PaymentOrderDetail;
import com.cku.oa.finance.service.PaymentChargingItemGroupService;
import com.cku.oa.finance.service.PaymentChargingItemService;
import com.cku.oa.finance.service.PaymentOrderService;
import com.cku.oa.sampling.entity.Sampling;
import com.cku.oa.sys.entity.user.Member;
import com.cku.oa.sys.service.user.MemberService;
import com.cku.restful.v1.finance.service.RestOrderService;
import com.thinkgem.jeesite.common.persistence.Page;
import com.thinkgem.jeesite.common.service.CrudService;
import com.thinkgem.jeesite.modules.sys.utils.UserUtils;

/**
 * 犬只DNA存档采样包补寄Service
 * 
 * @author Sunny
 * @version 2023-06-29
 */
@Service
@Transactional(readOnly = true)
public class DogDnaArchiveReissueSamplingService
		extends CrudService<DogDnaArchiveReissueSamplingDao, DogDnaArchiveReissueSampling> {

	@Autowired
	private DogDnaArchiveService dogDnaArchiveService;

	@Autowired
	private PaymentChargingItemGroupService paymentChargingItemGroupService;

	@Autowired
	private PaymentChargingItemService paymentChargingItemService;

	@Autowired
	private MemberService memberService;

	@Autowired
	private PaymentOrderService paymentOrderService;

	@Autowired
	private RestOrderService restOrderService;

	public DogDnaArchiveReissueSampling get(String id) {
		return super.get(id);
	}

	public List<DogDnaArchiveReissueSampling> findList(DogDnaArchiveReissueSampling dogDnaArchiveReissueSampling) {
		return super.findList(dogDnaArchiveReissueSampling);
	}

	public Page<DogDnaArchiveReissueSampling> findPage(Page<DogDnaArchiveReissueSampling> page,
			DogDnaArchiveReissueSampling dogDnaArchiveReissueSampling) {
		return super.findPage(page, dogDnaArchiveReissueSampling);
	}

	@Transactional(readOnly = false)
	public void save(DogDnaArchiveReissueSampling dogDnaArchiveReissueSampling) {
		super.save(dogDnaArchiveReissueSampling);
	}

	@Transactional(readOnly = false)
	public void delete(DogDnaArchiveReissueSampling dogDnaArchiveReissueSampling) {
		super.delete(dogDnaArchiveReissueSampling);
	}

	public List<Sampling> findReissueSamplingRecordByDnaArchiveId(String dnaArchiveId) {
		return dao.findReissueSamplingRecordByDnaArchiveId(dnaArchiveId);
	}

	public List<DogDnaArchiveReissueSampling> findByRunningNumber(String runningNumber) {
		return dao.findByRunningNumber(runningNumber);
	}

	/**
	 * 查询采样包绑定记录
	 * 
	 * @param dogDnaArchive
	 * @return
	 */
	public List<Sampling> findBindSamplingList(DogDnaArchive dogDnaArchive) {
		/**
		 * 由于新建采样包补寄记录表，故先注释原逻辑（根据采样包表查询方式）
		 */
//		String[] dogCodeArr = new String[] { dogDnaArchive.getPedigreeCertified(), dogDnaArchive.getBirthCode(),
//				dogDnaArchive.getOverseasPedigreeCertifiedLike() };
//		String[] newDogCodeArr = Arrays.stream(dogCodeArr).filter(dc -> StringUtils.isNotBlank(dc))
//				.toArray(String[]::new);
//		return samplingService.findBindRecordByDogCodes(Arrays.asList(newDogCodeArr));
		return this.findReissueSamplingRecordByDnaArchiveId(dogDnaArchive.getId());
	}

	private static final String BASE_PAYMENT_CHARGING_ITEM_GROUP_ID = "101";

	public List<PaymentChargingItem> getReissueSamplingItem(DogDnaArchive dogDnaArchive) {
		List<Sampling> bindSamplingList = this.findBindSamplingList(dogDnaArchive);
		if (bindSamplingList.size() >= 3) {
			throw new ZAException(ZAErrorCode.ZA_ERROR, "该犬只已有三次补寄记录，无法办理该业务！");
		}
		return getReissueSamplingItem(dogDnaArchive, bindSamplingList);
	}

	public List<PaymentChargingItem> getReissueSamplingItem(DogDnaArchive dogDnaArchive,
			List<Sampling> bindSamplingList) {
		// 拼接采样包补寄打包项ID
		// GROUP ID 0元|10111，20元|10121，280元|10131，400元|10132
		// 规则 前三位：101 为前缀，第四位为第几次补寄，第五位为本次补寄的分类
		// （因第一二次只有一种，默认1，第三次补寄分280和400，所以根据业务情况决定为1或2）
		StringBuilder chargingItemGroupId = new StringBuilder(BASE_PAYMENT_CHARGING_ITEM_GROUP_ID);
		int currentReissueTimes = 1;
		if (CollectionUtils.isNotEmpty(bindSamplingList)) {
			currentReissueTimes = currentReissueTimes + bindSamplingList.size();
		}
		chargingItemGroupId.append(currentReissueTimes);
		if (currentReissueTimes == 3) {
			// 第三次补寄，根据实际业务收费决定收费金额
			// 规则为，若业务收费280元，则本次收费280，若业务收费400或500时，本次收费400
			if (new BigDecimal(dogDnaArchive.getPrice()).compareTo(new BigDecimal("280")) == 0) {
				chargingItemGroupId.append("1");
			} else {
				chargingItemGroupId.append("2");
			}
		} else {
			chargingItemGroupId.append("1");
		}
		return paymentChargingItemGroupService
				.getPaymentChargingItemsByPaymentChargingItemGroupId(chargingItemGroupId.toString());
	}

	private PaymentOrder buildReissueSamplingPaymentOrder(DogDnaArchiveReissueSampling reissueSampling) {
		PaymentOrder order = new PaymentOrder();
		order.setMemberCode(reissueSampling.getPaymentMemberCode());
		order.setMemberName(reissueSampling.getPaymentMemberName());
		order.setAddTime(new Date());
		return order;
	}

	private List<PaymentOrderDetail> buildReissueSamplingPaymentOrderDetailList(PaymentOrder paymentOrder,
			DogDnaArchiveReissueSampling reissueSampling, DogDnaArchive dogDnaArchive) {
		List<PaymentOrderDetail> paymentOrderDetailList = new ArrayList<PaymentOrderDetail>();
		if (reissueSampling == null) {
			throw new ZAException(ZAErrorCode.ZA_VALID_FAILED, "采样包补寄记录有误!");
		} else {
			List<PaymentChargingItem> itemList = getReissueSamplingItem(dogDnaArchive);
			if (CollectionUtils.isEmpty(itemList)) {
				throw new ZAException(ZAErrorCode.ZA_ERROR, "采样包补寄收费项配置错误，请联系管理员处理！");
			}
			paymentOrderDetailList.addAll(itemList.stream().map(item -> {
				PaymentOrderDetail detail = new PaymentOrderDetail();
				detail.setAddTime(paymentOrder.getAddTime());
				detail.setProccessState("0");
				detail.setTotalPrice(item.getPrice());
				detail.setNum("1");
				detail.setPrice(item.getPrice());
				detail.setChargingItemId(item.getId());
				detail.setMemberCode(paymentOrder.getMemberCode());
				detail.setMemberName(paymentOrder.getMemberName());
				detail.setChargingItemName(item.getName());
				if (StringUtils.isNotBlank(item.getShortName()) && item.getShortName().contains("reissue")) {
					detail.setBusinessTable(item.getShortName().split("@")[0]);
					detail.setBusinessIds(reissueSampling.getId());
				}
				return detail;
			}).collect(Collectors.toList()));
		}
		return paymentOrderDetailList;
	}

	/**
	 * 补寄下单
	 * 
	 * @param dogDnaArchive
	 */
	@Transactional(readOnly = false)
	public void reissueSampling(DogDnaArchive dogDnaArchive) {
		DogDnaArchive dbDogDnaArchive = dogDnaArchiveService.get(dogDnaArchive.getId());
		// 创建采样包补寄记录
		DogDnaArchiveReissueSampling reissueSampling = new DogDnaArchiveReissueSampling();
		reissueSampling.setDnaArchiveId(dogDnaArchive.getId());
		reissueSampling.setOldSamplingCode(dbDogDnaArchive.getSamplingCode());
		reissueSampling.setSamplingCode(dogDnaArchive.getSamplingCode());
		reissueSampling.setSource(0);// 0 为DNA存档补寄业务补寄 1 为录订单补寄
		Member member = memberService.getByMemberCode(dogDnaArchive.getPaymentMemberCode());
		reissueSampling.setPaymentMemberCode(member.getMemberCode());
		reissueSampling.setPaymentMemberName(member.getName());
		reissueSampling.setOperator(UserUtils.getUser().getId());
		reissueSampling.setRemarks(dogDnaArchive.getRemarks());
		this.save(reissueSampling);
		// 设置订单基本字段
		PaymentOrder paymentOrder = buildReissueSamplingPaymentOrder(reissueSampling);
		// 构造订单详情列表
		List<PaymentOrderDetail> paymentOrderDetailList = buildReissueSamplingPaymentOrderDetailList(paymentOrder,
				reissueSampling, dogDnaArchive);
		// 计算订单总额
		BigDecimal orderTotal = paymentOrderDetailList.stream().map(od -> {
			return new BigDecimal(od.getPrice());
		}).reduce(BigDecimal.ZERO, BigDecimal::add);
		// 保存订单
		paymentOrderService.saveOrder(paymentOrder, orderTotal.doubleValue(), paymentOrder.getAddTime());
		// 回写订单详情的OrderCode字段
		paymentOrderDetailList.forEach(detail -> {
			detail.setOrderCode(paymentOrder.getOrderCode());
		});
		// 更新记录订单号及金额
		reissueSampling.setRunningNumber(paymentOrder.getOrderCode());
		reissueSampling.setPrice(orderTotal.doubleValue());
		this.save(reissueSampling);
		// 保存订单详情
		paymentOrderService.addOrderDetailList(paymentOrderDetailList);
		paymentOrderService.balPay(paymentOrder);
		// 调用公共订单支付回调
		restOrderService.afterPay(paymentOrder);
	}

	@Transactional(readOnly = false)
	public void orderSave(DogDnaArchive dogDnaArchive) {
		DogDnaArchive dbDogDnaArchive = dogDnaArchiveService.get(dogDnaArchive.getId());
		// 创建采样包补寄记录
		DogDnaArchiveReissueSampling reissueSampling = new DogDnaArchiveReissueSampling();
		reissueSampling.setDnaArchiveId(dogDnaArchive.getId());
		reissueSampling.setOldSamplingCode(dbDogDnaArchive.getSamplingCode());
		reissueSampling.setSamplingCode(dogDnaArchive.getSamplingCode());
		reissueSampling.setSource(1);// 0 为DNA存档补寄业务补寄 1 为录订单补寄
		Member member = memberService.getByMemberCode(dbDogDnaArchive.getPaymentMemberCode());
		reissueSampling.setPaymentMemberCode(member.getMemberCode());
		reissueSampling.setPaymentMemberName(member.getName());
		reissueSampling.setOperator(UserUtils.getUser().getId());
		reissueSampling.setRemarks(dogDnaArchive.getRemarks());
		PaymentOrder paymentOrder = paymentOrderService.getByOrderCode(dogDnaArchive.getRunningNumber());
		// 更新记录订单号及金额
		reissueSampling.setRunningNumber(paymentOrder.getOrderCode());
		reissueSampling.setPrice(Double.valueOf(paymentOrder.getTotalPrice()));
		reissueSampling.setPaymentMemberCode(paymentOrder.getMemberCode());
		reissueSampling.setPaymentState(paymentOrder.getPaymentState());
		reissueSampling.setPaymentTime(paymentOrder.getPaymentTime());
		this.save(reissueSampling);
		PaymentChargingItem chargingItem = paymentChargingItemService.get(dogDnaArchive.getPaymentChargingItemId());
		// 保存订单详情
		paymentOrderService.updateBusinessInfo(paymentOrder.getOrderCode(), dogDnaArchive.getOrderDetailId(),
				chargingItem.getId(), reissueSampling.getId(), chargingItem.getBusinessTable());
		dogDnaArchiveService.afterPayReissueSampling(reissueSampling.getId(), dogDnaArchive.getPaymentChargingItemId());
	}

}